<?php

session_start();

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/shopify-oauth.php';
require_once __DIR__ . '/includes/shopify-customer.php';
require_once __DIR__ . '/includes/tridentity-api.php';

logMessage("=== CALLBACK HANDLER STARTED ===");
logMessage("GET parameters: " . json_encode($_GET));

// Fix malformed URL with multiple question marks
if (isset($_GET['shop']) && (strpos($_GET['shop'], '?token=') !== false || strpos($_GET['shop'], '?') !== false)) {
    logMessage("Detected malformed URL parameter in shop: " . $_GET['shop'], 'DEBUG');

    // Handle cases where shop parameter contains additional parameters
    $shopParam = $_GET['shop'];

    // Check if there's a question mark in the shop parameter
    if (strpos($shopParam, '?') !== false) {
        $parts = explode('?', $shopParam, 2);
        $_GET['shop'] = $parts[0]; // First part is the actual shop domain

        // Parse the remaining part as query string
        if (isset($parts[1])) {
            parse_str($parts[1], $additionalParams);
            foreach ($additionalParams as $key => $value) {
                if (!isset($_GET[$key])) {
                    $_GET[$key] = $value;
                }
            }
        }
    }

    logMessage("After fix - Shop: " . $_GET['shop'] . ", Token: " . ($_GET['token'] ?? 'NOT SET'), 'DEBUG');
}

// Also check if token might be in a different malformed format
if (!isset($_GET['token']) && isset($_GET['shop'])) {
    $shopParam = $_GET['shop'];
    if (preg_match('/token=([^&]+)/', $shopParam, $matches)) {
        $_GET['token'] = $matches[1];
        // Clean the shop parameter
        $_GET['shop'] = preg_replace('/\?token=[^&]+/', '', $shopParam);
        logMessage("Extracted token from shop parameter - Shop: " . $_GET['shop'] . ", Token: " . $_GET['token'], 'DEBUG');
    }
}

$shop = $_GET['shop'] ?? '';
$code = $_GET['code'] ?? '';
$state = $_GET['state'] ?? '';
$host = $_GET['host'] ?? '';
$token = $_GET['token'] ?? '';

// Handle OAuth callback (installation)
if ($code && $state) {
    logMessage("Processing OAuth callback");

    // Verify state to prevent CSRF
    if (!isset($_SESSION['oauth_state']) || $_SESSION['oauth_state'] !== $state) {
        logMessage("OAuth state mismatch", 'ERROR');
        die('Invalid OAuth state');
    }

    $shop = $_SESSION['install_shop'] ?? $shop;
    $host = $_SESSION['install_host'] ?? $host;

    unset($_SESSION['oauth_state']);
    unset($_SESSION['install_shop']);
    unset($_SESSION['install_host']);

    if (!$shop) {
        logMessage("No shop parameter in OAuth callback", 'ERROR');
        die('Shop parameter is required');
    }

    $shop = cleanShopDomain($shop);

    try {
        logMessage("Exchanging authorization code for access token", 'INFO', $shop);

        $accessToken = exchangeCodeForAccessToken($shop, $code);

        if ($accessToken && TokenStorage::storeAccessToken($shop, $accessToken)) {
            logMessage("Access token saved successfully", 'INFO', $shop);

            $redirectUrl = "index.php?shop=" . urlencode($shop) . "&installed=1";
            if ($host) {
                $redirectUrl .= "&host=" . urlencode($host);
            }
            header("Location: " . $redirectUrl);
            exit;
        }

        throw new Exception('Failed to obtain access token from Shopify');
    } catch (Exception $e) {
        logMessage("OAuth callback error: " . $e->getMessage(), 'ERROR', $shop);
        echo "Installation error: " . $e->getMessage();
        exit;
    }
}

// Handle Tridentity callback (login)
else if ($token && $shop) {

    logMessage("Processing Tridentity callback", 'INFO', $shop);

    $shop = cleanShopDomain($shop);

    try {
        // Check if app is installed
        if (!TokenStorage::isInstalled($shop)) {
            throw new Exception('App not installed. Please install the app first.');
        }

        // Check if Tridentity settings are configured
        $settings = AppSettings::getSettings($shop);
        if (empty($settings['tridentity_app_id']) || empty($settings['tridentity_api_key'])) {
            throw new Exception('Tridentity settings are not configured. Please configure them in the app settings.');
        }

        // DEBUG: Check token and API access
        logMessage("DEBUG: Checking access token and API access", 'DEBUG', $shop);
        $accessToken = TokenStorage::getAccessToken($shop);

        if (!$accessToken) {
            throw new Exception('No access token found. Please reinstall the app.');
        }

        logMessage("DEBUG: Access token retrieved: " . substr($accessToken, 0, 20) . "...", 'DEBUG', $shop);

        // Test API access
        if (!testShopifyAPI($shop, $accessToken)) {
            throw new Exception('Shopify API access denied. The access token may be invalid. Please reinstall the app.');
        }

        logMessage("DEBUG: Shopify API access confirmed", 'DEBUG', $shop);

        logMessage("Step 1: Verifying Tridentity token", 'INFO', $shop);

        $identityToken = verifyTridentityToken($token);

        if (!$identityToken) {
            throw new Exception('Invalid authentication token received from Tridentity');
        }

        logMessage("Step 2: Fetching user profile", 'INFO', $shop);
        $userProfile = getTridentityUserProfile($identityToken);

        if (!$userProfile) {
            throw new Exception('Failed to retrieve user profile from Tridentity');
        }

        logMessage("Tridentity profile data: " . json_encode($userProfile), 'DEBUG', $shop);

        $userData = $userProfile['data']['user'] ?? [];
        $accountData = $userProfile['data']['account'] ?? [];

        // Generate unique email if not provided by Tridentity
        $originalEmail = $accountData['email'] ?? '';
        $phone = $accountData['phone'] ?? '';
        $phoneCode = $accountData['phoneCode'] ?? '91';
        $membershipId = $userData['membershipId'] ?? uniqid();
        $userId = $userData['userId'] ?? uniqid();

        if (empty($originalEmail)) {
            // Generate unique email based on available data
            if (!empty($phone)) {
                $email = $phoneCode . $phone . '@tridentity.test';
            } else {
                $email = 'user_' . $userId . '_' . substr($membershipId, -6) . '@tridentity.test';
            }
            logMessage("Generated unique email for user: $email", 'INFO', $shop);
        } else {
            $email = $originalEmail;
        }

        // Generate proper names if empty
        $firstName = !empty(trim($userData['firstName'] ?? '')) ? trim($userData['firstName']) : 'Tridentity';
        $lastName = !empty(trim($userData['lastName'] ?? '')) ? trim($userData['lastName']) : 'User';

        // If both names are default, use membership ID for uniqueness
        if ($firstName === 'Tridentity' && $lastName === 'User') {
            $firstName = 'Tridentity';
            $lastName = 'User_' . substr($membershipId, -6);
        }

        logMessage("Step 3: Processing user - Email: $email, Name: $firstName $lastName, Phone: $phone, Membership: $membershipId", 'INFO', $shop);

        // Step 4: Find or create customer in Shopify
        logMessage("Step 4: Finding or creating customer in Shopify", 'INFO', $shop);

        // Debug customer data
        logMessage("Customer creation data - Email: $email, First: $firstName, Last: $lastName, Phone: $phone", 'DEBUG', $shop);

        $customer = findOrCreateCustomer($shop, $email, $firstName, $lastName, $phone);

        // Temporary debug
        logMessage("CUSTOMER CREATION RESULT: " . json_encode($customer), 'DEBUG', $shop);

        if (!$customer || !isset($customer['id'])) {
            throw new Exception('Failed to find or create customer in Shopify. Customer object: ' . json_encode($customer));
        }

        logMessage("Step 5: Customer processed successfully - ID: " . $customer['id'], 'INFO', $shop);

        // Step 6: Generate login URL with activation
        logMessage("Step 6: Generating login URL with activation", 'INFO', $shop);

        $loginUrl = generateCustomerLoginUrl($shop, $customer['id'], $email);

        if ($loginUrl) {
            logMessage("Step 7: Successfully generated login URL", 'INFO', $shop);
            logMessage("Login URL: $loginUrl", 'DEBUG', $shop);

            // Step 8: Redirect customer to login URL
            logMessage("Step 8: Redirecting customer to login URL", 'INFO', $shop);
            header("Location: $loginUrl");
            exit;
        } else {
            logMessage("Step 7: Failed to generate login URL, using fallback", 'WARNING', $shop);

            // Fallback: Redirect to customer account page
            $accountUrl = "https://$shop/account?tridentity_activated=1";
            logMessage("Step 8: Fallback - redirecting to account page: $accountUrl", 'INFO', $shop);
            header("Location: $accountUrl");
            exit;
        }
    } catch (Exception $e) {
        logMessage("CALLBACK ERROR: " . $e->getMessage(), 'ERROR', $shop);

        // Redirect to Shopify login page with error message
        $errorUrl = "https://$shop/account/login?error=tridentity_login_failed&message=" . urlencode($e->getMessage());
        header("Location: $errorUrl");
        exit;
    }
} else {
    logMessage("Invalid callback - no code or token provided", 'ERROR');
    echo "Invalid callback parameters";
    exit;
}
