<?php

// Enable error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

// Load environment variables from config/.env
if (file_exists(__DIR__ . '/.env')) {
    $env = parse_ini_file(__DIR__ . '/.env');
    foreach ($env as $key => $value) {
        $_ENV[$key] = $value;
    }
}

// Define base paths and URLs
define('APP_BASE_URL', 'https://qztbox.com/shopify/tridentity-login-app');
define('APP_URL', APP_BASE_URL);
define('APP_CALLBACK_URL', APP_BASE_URL . '/callback.php');
define('APP_ROOT', dirname(dirname(__FILE__)));

// Shopify OAuth Configuration
define('SHOPIFY_API_KEY', $_ENV['SHOPIFY_API_KEY'] ?? '');
define('SHOPIFY_API_SECRET', $_ENV['SHOPIFY_API_SECRET'] ?? '');
define('SHOPIFY_SCOPES', 'read_customers,write_customers,read_customer_tags,write_customer_tags');
define('SHOPIFY_API_VERSION', '2024-07');

// Storage for access tokens
define('STORAGE_PATH', APP_ROOT . '/storage');

if (!is_dir(STORAGE_PATH)) {
    mkdir(STORAGE_PATH, 0755, true);
}

if (!is_dir(STORAGE_PATH . '/tokens')) {
    mkdir(STORAGE_PATH . '/tokens', 0755, true);
}

if (!is_dir(STORAGE_PATH . '/logs')) {
    mkdir(STORAGE_PATH . '/logs', 0755, true);
}

if (!is_dir(STORAGE_PATH . '/settings')) {
    mkdir(STORAGE_PATH . '/settings', 0755, true);
}

// Simple logging
function logMessage($message, $level = 'INFO', $shop = '')
{
    $timestamp = date('Y-m-d H:i:s');
    $logFile = $shop ? STORAGE_PATH . "/logs/app_$shop.log" : STORAGE_PATH . '/logs/app.log';

    if (!is_dir(dirname($logFile))) {
        mkdir(dirname($logFile), 0755, true);
    }

    $logMessage = "[$timestamp] [$level] $message" . PHP_EOL;
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

// HMAC verification
function verifyHmac($params)
{
    if (!isset($params['hmac'])) {
        logMessage("HMAC verification failed: No HMAC parameter provided");
        return false;
    }

    $hmac = $params['hmac'];
    unset($params['hmac']);

    // Also remove signature if present (some requests use signature instead of hmac)
    if (isset($params['signature'])) {
        unset($params['signature']);
    }

    // Sort parameters
    ksort($params);

    // Build message
    $message = http_build_query($params);

    // Calculate HMAC
    $calculatedHmac = hash_hmac('sha256', $message, SHOPIFY_API_SECRET);

    // Use hash_equals for timing attack protection
    $isValid = hash_equals($hmac, $calculatedHmac);

    if (!$isValid) {
        logMessage("HMAC verification failed: Calculated: $calculatedHmac, Received: $hmac");
        logMessage("Parameters: " . json_encode($params));
    } else {
        logMessage("HMAC verification successful");
    }

    return $isValid;
}

// Shop domain validation
function cleanShopDomain($shop)
{
    if (empty($shop)) {
        return '';
    }
    $shop = preg_replace('#^https?://#', '', $shop);
    $shop = preg_replace('#(/.*)?$#', '', $shop);
    return $shop;
}

function verifyShopDomain($shop)
{
    return !empty($shop) && preg_match('/\.myshopify\.com$/', $shop);
}

// Simple token storage
class TokenStorage
{
    public static function storeAccessToken($shop, $accessToken)
    {
        $tokenFile = STORAGE_PATH . "/tokens/" . preg_replace('/[^a-zA-Z0-9_\-]/', '_', $shop) . ".json";

        $tokenData = [
            'access_token' => $accessToken,
            'shop_domain' => $shop,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];

        $result = file_put_contents($tokenFile, json_encode($tokenData, JSON_PRETTY_PRINT));

        if ($result !== false) {
            logMessage("Access token stored successfully for shop: $shop", 'INFO');
        } else {
            logMessage("Failed to store access token for shop: $shop", 'ERROR');
        }

        return $result !== false;
    }

    public static function getAccessToken($shop)
    {
        $tokenFile = STORAGE_PATH . "/tokens/" . preg_replace('/[^a-zA-Z0-9_\-]/', '_', $shop) . ".json";

        if (file_exists($tokenFile)) {
            $tokenData = json_decode(file_get_contents($tokenFile), true);
            $token = $tokenData['access_token'] ?? null;

            if ($token) {
                logMessage("Retrieved access token for shop: $shop", 'DEBUG');
            } else {
                logMessage("No access token found in token file for shop: $shop", 'DEBUG');
            }

            return $token;
        }

        logMessage("Token file not found for shop: $shop", 'DEBUG');
        return null;
    }

    public static function isInstalled($shop)
    {
        $token = self::getAccessToken($shop);
        $isInstalled = $token !== null;
        logMessage("Installation check for $shop: " . ($isInstalled ? 'INSTALLED' : 'NOT INSTALLED'), 'DEBUG');
        return $isInstalled;
    }

    public static function clearInstallation($shop)
    {
        $tokenFile = STORAGE_PATH . "/tokens/" . preg_replace('/[^a-zA-Z0-9_\-]/', '_', $shop) . ".json";
        if (file_exists($tokenFile)) {
            return unlink($tokenFile);
        }
        return true;
    }
}

// App Settings Storage
class AppSettings
{
    public static function getSettings($shop)
    {
        $settingsFile = STORAGE_PATH . "/settings/" . preg_replace('/[^a-zA-Z0-9_\-]/', '_', $shop) . ".json";

        if (file_exists($settingsFile)) {
            $settings = json_decode(file_get_contents($settingsFile), true);
            return $settings ?: [];
        }

        // Return default settings
        return [
            'tridentity_app_id' => '',
            'tridentity_api_key' => '',
            'tridentity_api_url' => 'https://triapp-api-staging.tribox.me',
            'tridentity_outer_url' => 'https://tridentity-outer-staging.tribox.me',
            'updated_at' => null
        ];
    }

    public static function saveSettings($shop, $settings)
    {
        $settingsDir = STORAGE_PATH . "/settings";
        if (!is_dir($settingsDir)) {
            mkdir($settingsDir, 0755, true);
        }

        $settingsFile = $settingsDir . "/" . preg_replace('/[^a-zA-Z0-9_\-]/', '_', $shop) . ".json";

        $settingsData = [
            'tridentity_app_id' => $settings['tridentity_app_id'] ?? '',
            'tridentity_api_key' => $settings['tridentity_api_key'] ?? '',
            'tridentity_api_url' => $settings['tridentity_api_url'] ?? 'https://triapp-api-staging.tribox.me',
            'tridentity_outer_url' => $settings['tridentity_outer_url'] ?? 'https://tridentity-outer-staging.tribox.me',
            'updated_at' => date('Y-m-d H:i:s')
        ];

        $result = file_put_contents($settingsFile, json_encode($settingsData, JSON_PRETTY_PRINT));

        if ($result !== false) {
            logMessage("Settings saved successfully for shop: $shop", 'INFO');
        } else {
            logMessage("Failed to save settings for shop: $shop", 'ERROR');
        }

        return $result !== false;
    }

    public static function validateSettings($settings)
    {
        $errors = [];

        if (empty($settings['tridentity_app_id'])) {
            $errors[] = 'Tridentity App ID is required';
        }

        if (empty($settings['tridentity_api_key'])) {
            $errors[] = 'Tridentity API Key is required';
        }

        if (empty($settings['tridentity_api_url'])) {
            $errors[] = 'Tridentity API URL is required';
        }

        return $errors;
    }
}
