<?php

/**
 * Tridentity API Handler
 */

/**
 * Get Tridentity settings for shop
 */
function getTridentitySettings($shop)
{
    return AppSettings::getSettings($shop);
}

/**
 * Verify Tridentity authentication token
 */
function verifyTridentityToken($authToken)
{
    $shop = $_GET['shop'] ?? '';
    $settings = getTridentitySettings($shop);

    if (empty($settings['tridentity_app_id']) || empty($settings['tridentity_api_key']) || empty($settings['tridentity_api_url'])) {
        logMessage("Tridentity settings not configured for shop: $shop", 'ERROR');
        return false;
    }

    $url = $settings['tridentity_api_url'] . '/external/auth/token/verify';

    logMessage("Tridentity Token Verification - URL: $url", 'DEBUG');
    logMessage("Tridentity Token Verification - Auth Token: " . substr($authToken, 0, 10) . "...", 'DEBUG');

    $postData = json_encode(['authToken' => $authToken]);

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $postData,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'api-key: ' . $settings['tridentity_api_key'],
            'app-id: ' . $settings['tridentity_app_id']
        ],
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 30
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    logMessage("Tridentity token verification response: HTTP $httpCode - $response", 'INFO');

    if ($curlError) {
        logMessage("Tridentity token verification CURL error: $curlError", 'ERROR');
        return false;
    }

    if ($httpCode === 200 || $httpCode === 201) {
        $data = json_decode($response, true);
        $identityToken = $data['data']['user']['identityToken'] ?? false;

        if ($identityToken) {
            logMessage("Successfully obtained identity token", 'INFO');
        } else {
            logMessage("No identity token in response", 'ERROR');
        }

        return $identityToken;
    }

    logMessage("Tridentity token verification failed with HTTP $httpCode", 'ERROR');
    return false;
}

/**
 * Get user profile from Tridentity
 */
function getTridentityUserProfile($identityToken)
{
    $shop = $_GET['shop'] ?? '';
    $settings = getTridentitySettings($shop);

    if (empty($settings['tridentity_app_id']) || empty($settings['tridentity_api_key']) || empty($settings['tridentity_api_url'])) {
        logMessage("Tridentity settings not configured for shop: $shop", 'ERROR');
        return false;
    }

    $url = $settings['tridentity_api_url'] . 's/external/user/profile';

    logMessage("Tridentity Profile Fetch - URL: $url", 'DEBUG');
    logMessage("Tridentity Profile Fetch - Identity Token: " . substr($identityToken, 0, 10) . "...", 'DEBUG');

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'app-id: ' . $settings['tridentity_app_id'],
            'api-key: ' . $settings['tridentity_api_key'],
            'identity-token: ' . $identityToken,
            'Content-Type: application/json'
        ],
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 30
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    logMessage("Tridentity profile fetch response: HTTP $httpCode", 'INFO');

    if ($curlError) {
        logMessage("Tridentity profile fetch CURL error: $curlError", 'ERROR');
        return false;
    }

    if ($httpCode === 200) {
        $data = json_decode($response, true);
        logMessage("Successfully retrieved user profile from Tridentity", 'INFO');
        return $data;
    }

    logMessage("Tridentity profile fetch failed with HTTP $httpCode: $response", 'ERROR');
    return false;
}

/**
 * Generate Tridentity login URL
 */
function generateTridentityLoginUrl($shop)
{
    $settings = getTridentitySettings($shop);

    if (empty($settings['tridentity_app_id']) || empty($settings['tridentity_outer_url'])) {
        logMessage("Tridentity settings not configured for shop: $shop", 'ERROR');
        return '#';
    }

    $callbackUrl = APP_CALLBACK_URL . '?shop=' . urlencode($shop);
    $loginUrl = $settings['tridentity_outer_url'] . '/oauth/login?app_id=' . $settings['tridentity_app_id'] . '&redirect=' . urlencode($callbackUrl);

    logMessage("Generated Tridentity login URL: $loginUrl", 'DEBUG');

    return $loginUrl;
}
