<?php
session_start();
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/shopify-oauth.php';
require_once __DIR__ . '/includes/tridentity-api.php';

$shop = $_GET['shop'] ?? '';
$host = $_GET['host'] ?? '';

if (!$shop) {
    die('Shop parameter is required. Please access this app through the Shopify Admin.');
}

$shop = cleanShopDomain($shop);

// Verify shop domain
if (!verifyShopDomain($shop)) {
    die('Invalid Shopify store domain. Please use: your-store.myshopify.com');
}

// Verify installation and token validity
$isInstalled = TokenStorage::isInstalled($shop);
$accessToken = TokenStorage::getAccessToken($shop);
$tokenValid = false;

if ($isInstalled && $accessToken) {
    // Test if token is still valid
    $url = "https://$shop/admin/api/" . SHOPIFY_API_VERSION . "/shop.json";
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'X-Shopify-Access-Token: ' . $accessToken,
            'Content-Type: application/json'
        ],
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 10
    ]);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $tokenValid = ($httpCode === 200);

    if (!$tokenValid) {
        logMessage("Stored token is invalid - clearing installation", 'WARNING', $shop);
        TokenStorage::clearInstallation($shop);
        $isInstalled = false;
    }
}

// If not installed or token invalid, redirect to install
if (!$isInstalled || !$tokenValid) {
    logMessage("App not installed or token invalid - redirecting to install", 'INFO', $shop);

    // Use install.php to handle the installation with proper HMAC
    $installUrl = "install.php?shop=" . urlencode($shop);
    if ($host) {
        $installUrl .= "&host=" . urlencode($host);
    }

    header("Location: " . $installUrl);
    exit;
}

// App is properly installed - show the main app interface
logMessage("App loaded successfully for shop: $shop", 'INFO', $shop);

// Handle installation success message
$success_message = '';
if (isset($_GET['installed'])) {
    $success_message = "App installed successfully! You can now test the Tridentity login flow.";
}

// Check settings configuration
$settings = AppSettings::getSettings($shop);
$settingsConfigured = !empty($settings['tridentity_app_id']) && !empty($settings['tridentity_api_key']);
?>

<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tridentity Login App</title>
    <link rel="stylesheet" href="assets/style.css">
</head>

<body>
    <div class="app-container">
        <!-- Header -->
        <div class="card">
            <div class="card-header">
                <div class="header-content">
                    <h1>Tridentity Login</h1>
                    <div class="navigation">
                        <a href="index.php?shop=<?= urlencode($shop) ?>" class="nav-btn active">Dashboard</a>
                        <a href="settings.php?shop=<?= urlencode($shop) ?>" class="nav-btn">Settings</a>
                        <a href="install.php?shop=<?= urlencode($shop) ?>" class="nav-btn" onclick="return confirm('Reinstall the app? This will clear current settings and generate a new access token.')">Reinstall</a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Success Message -->
        <?php if ($success_message): ?>
            <div class="alert alert-success"><?= htmlspecialchars($success_message) ?></div>
        <?php endif; ?>

        <!-- Settings Warning -->
        <?php if (!$settingsConfigured): ?>
            <div class="alert alert-warning">
                Tridentity settings are not configured. <a href="settings.php?shop=<?= urlencode($shop) ?>">Click here to configure them</a> before testing the login flow.
            </div>
        <?php endif; ?>

        <!-- Main Dashboard -->
        <div class="card">
            <div class="card-body">
                <h2>Welcome to Tridentity Login App</h2>
                <p>Enable seamless customer authentication through Tridentity.</p>

                <div class="status-section">
                    <h3>App Status</h3>
                    <div class="status-grid">
                        <div class="status-item status-success">
                            ✓ App Installed
                        </div>
                        <div class="status-item status-success">
                            ✓ Shopify Connected
                        </div>
                        <div class="status-item <?= $settingsConfigured ? 'status-success' : 'status-warning' ?>">
                            <?= $settingsConfigured ? '✓ Settings Configured' : '⚙ Settings Needed' ?>
                        </div>
                    </div>
                </div>

                <div style="margin: 24px 0;">
                    <a href="<?= generateTridentityLoginUrl($shop) ?>"
                        class="btn btn-primary"
                        <?= !$settingsConfigured ? 'disabled onclick="return false;"' : '' ?>>
                        Test Tridentity Login
                    </a>
                    <?php if (!$settingsConfigured): ?>
                        <p style="color: #8a6500; margin-top: 8px;">Please configure Tridentity settings first.</p>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Setup Instructions -->
        <div class="card">
            <div class="card-body">
                <h3>Setup Instructions</h3>
                <ol class="instructions-list">
                    <li>Configure your Tridentity credentials in the Settings page</li>
                    <li>Click "Test Tridentity Login" to test the complete flow</li>
                    <li>You'll be redirected to Tridentity login page</li>
                    <li>After successful login, you'll be redirected back</li>
                    <li>The system will automatically create/find your customer in Shopify</li>
                    <li>You'll be automatically logged into your Shopify customer account</li>
                </ol>
            </div>
        </div>

        <!-- Integration Code -->
        <div class="card">
            <div class="card-body">
                <h3>Integration Code for Your Theme</h3>
                <p>Add this code to your Shopify theme where you want the login button to appear:</p>

                <div class="code-block">
                    <pre><code>&lt;div class="tridentity-login-section"&gt;
    &lt;a href="<?= generateTridentityLoginUrl($shop) ?>" class="btn btn-tdt-login"&gt;
        Login with Tridentity
    &lt;/a&gt;
&lt;/div&gt;</code></pre>
                </div>
            </div>
        </div>

        <!-- Debug Information -->
        <div class="card">
            <div class="card-body">
                <h3>Debug Information</h3>
                <div class="debug-info">
                    <h4>Connection Status</h4>
                    <div class="debug-grid">
                        <span class="debug-label">Shop:</span>
                        <span class="debug-value"><?= htmlspecialchars($shop) ?></span>

                        <span class="debug-label">Token Status:</span>
                        <span class="debug-value" style="color: #2e5e2e;">✓ Valid</span>

                        <span class="debug-label">Token Preview:</span>
                        <span class="debug-value"><?= substr($accessToken, 0, 20) ?>...</span>

                        <span class="debug-label">Settings Status:</span>
                        <span class="debug-value" style="color: <?= $settingsConfigured ? '#2e5e2e' : '#8a6500' ?>;">
                            <?= $settingsConfigured ? 'Configured' : 'Not Configured' ?>
                        </span>

                        <span class="debug-label">App Base URL:</span>
                        <span class="debug-value"><?= APP_BASE_URL ?></span>

                        <span class="debug-label">Callback URL:</span>
                        <span class="debug-value"><?= APP_CALLBACK_URL ?></span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>

</html>