<?php
session_start();
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/shopify-oauth.php';
require_once __DIR__ . '/includes/tridentity-api.php';

$shop = $_GET['shop'] ?? '';
$shop = cleanShopDomain($shop);

if (!$shop) {
    die('Shop parameter is required');
}

// Verify installation
if (!TokenStorage::isInstalled($shop)) {
    header("Location: install.php?shop=" . urlencode($shop));
    exit;
}

// Handle form submission
$success_message = '';
$error_message = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $settings = [
        'tridentity_app_id' => trim($_POST['tridentity_app_id'] ?? ''),
        'tridentity_api_key' => trim($_POST['tridentity_api_key'] ?? ''),
        'tridentity_api_url' => trim($_POST['tridentity_api_url'] ?? ''),
        'tridentity_outer_url' => trim($_POST['tridentity_outer_url'] ?? '')
    ];

    // Validate settings
    $errors = AppSettings::validateSettings($settings);

    if (empty($errors)) {
        if (AppSettings::saveSettings($shop, $settings)) {
            $success_message = 'Settings saved successfully!';

            // Test the connection
            $testResult = testTridentityConnection($shop, $settings);
            if ($testResult['success']) {
                $success_message .= ' Tridentity connection test passed!';
            } else {
                $error_message = 'Settings saved but Tridentity connection failed: ' . $testResult['message'];
            }
        } else {
            $error_message = 'Failed to save settings.';
        }
    } else {
        $error_message = implode('<br>', $errors);
    }
}

// Get current settings
$currentSettings = AppSettings::getSettings($shop);

/**
 * Test Tridentity connection with provided settings
 */
function testTridentityConnection($shop, $settings)
{
    // Use a simple API call to test the connection
    $testUrl = $settings['tridentity_api_url'] . '/v1/external/auth/token/verify';

    $postData = json_encode(['authToken' => 'test']);

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $testUrl,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $postData,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'api-key: ' . $settings['tridentity_api_key'],
            'app-id: ' . $settings['tridentity_app_id']
        ],
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 10
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($httpCode === 404) {
        // 404 might mean the endpoint doesn't exist but connection is working
        return ['success' => true, 'message' => 'Connection successful (endpoint may require valid token)'];
    } elseif ($httpCode === 401 || $httpCode === 403) {
        return ['success' => false, 'message' => 'Authentication failed - check App ID and API Key'];
    } elseif ($httpCode >= 200 && $httpCode < 300) {
        return ['success' => true, 'message' => 'Connection successful'];
    } else {
        return ['success' => false, 'message' => "HTTP $httpCode - Check API URL and credentials"];
    }
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tridentity Settings</title>
    <link rel="stylesheet" href="assets/style.css">
</head>

<body>
    <div class="app-container">
        <!-- Header -->
        <div class="card">
            <div class="card-header">
                <div class="header-content">
                    <h1>Tridentity Settings</h1>
                    <div class="navigation">
                        <a href="index.php?shop=<?= urlencode($shop) ?>" class="nav-btn">Dashboard</a>
                        <a href="settings.php?shop=<?= urlencode($shop) ?>" class="nav-btn active">Settings</a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Messages -->
        <?php if ($success_message): ?>
            <div class="alert alert-success"><?= $success_message ?></div>
        <?php endif; ?>

        <?php if ($error_message): ?>
            <div class="alert alert-error"><?= $error_message ?></div>
        <?php endif; ?>

        <!-- Settings Form -->
        <div class="card">
            <div class="card-body">
                <h2>Tridentity API Configuration</h2>
                <p>Configure your Tridentity API credentials to enable customer authentication.</p>

                <form method="POST" action="settings.php?shop=<?= urlencode($shop) ?>">
                    <div class="form-group">
                        <label class="form-label" for="tridentity_app_id">Tridentity App ID *</label>
                        <input type="text"
                            id="tridentity_app_id"
                            name="tridentity_app_id"
                            value="<?= htmlspecialchars($currentSettings['tridentity_app_id']) ?>"
                            class="form-input"
                            placeholder="Enter your Tridentity App ID"
                            required>
                        <div class="form-help">Your Tridentity application identifier</div>
                    </div>

                    <div class="form-group">
                        <label class="form-label" for="tridentity_api_key">Tridentity API Key *</label>
                        <input type="password"
                            id="tridentity_api_key"
                            name="tridentity_api_key"
                            value="<?= htmlspecialchars($currentSettings['tridentity_api_key']) ?>"
                            class="form-input"
                            placeholder="Enter your Tridentity API Key"
                            required>
                        <div class="form-help">Your Tridentity API secret key</div>
                    </div>

                    <div class="form-group">
                        <label class="form-label" for="tridentity_api_url">Tridentity API URL *</label>
                        <input type="url"
                            id="tridentity_api_url"
                            name="tridentity_api_url"
                            value="<?= htmlspecialchars($currentSettings['tridentity_api_url']) ?>"
                            class="form-input"
                            placeholder="https://api.tridentity.com"
                            required>
                        <div class="form-help">Base URL for Tridentity API (e.g., https://triapp-api-staging.tribox.me)</div>
                    </div>

                    <div class="form-group">
                        <label class="form-label" for="tridentity_outer_url">Tridentity Outer URL *</label>
                        <input type="url"
                            id="tridentity_outer_url"
                            name="tridentity_outer_url"
                            value="<?= htmlspecialchars($currentSettings['tridentity_outer_url']) ?>"
                            class="form-input"
                            placeholder="https://tridentity-outer-staging.tribox.me"
                            required>
                        <div class="form-help">URL for Tridentity login page (e.g., https://tridentity-outer-staging.tribox.me)</div>
                    </div>

                    <div class="form-actions">
                        <button type="submit" class="btn">Save Settings</button>
                        <a href="index.php?shop=<?= urlencode($shop) ?>" class="btn btn-secondary">Cancel</a>
                    </div>
                </form>

                <div class="test-section">
                    <h3>Test Connection</h3>
                    <p>After saving settings, you can test the Tridentity connection by using the "Test Tridentity Login" button on the dashboard.</p>
                </div>
            </div>
        </div>

        <!-- Instructions -->
        <div class="card">
            <div class="card-body">
                <h3>How to Get Tridentity Credentials</h3>
                <ol style="margin-left: 20px; color: #6d7175;">
                    <li>Contact Tridentity to get your App ID and API Key</li>
                    <li>Use the provided API URLs or ask for the correct endpoints</li>
                    <li>Save the settings and test the connection</li>
                    <li>If connection fails, verify your credentials with Tridentity support</li>
                </ol>
            </div>
        </div>
    </div>
</body>

</html>