<?php

session_start();
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/shopify-oauth.php';

logMessage("=== APP.PH ACCESSED ===");
logMessage("GET parameters: " . json_encode($_GET));

$shop = $_GET['shop'] ?? '';
$hmac = $_GET['hmac'] ?? '';
$timestamp = $_GET['timestamp'] ?? '';
$code = $_GET['code'] ?? '';
$host = $_GET['host'] ?? '';
$id_token = $_GET['id_token'] ?? '';

// If no shop provided, show install form
if (!$shop) {
    showInstallForm();
    exit;
}

$shop = cleanShopDomain($shop);
if (!verifyShopDomain($shop)) {
    die('Invalid Shopify store domain: ' . htmlspecialchars($shop));
}

// Verify HMAC if present
if ($hmac && !verifyHmac($_GET)) {
    die('Invalid HMAC signature. This request may not be from Shopify.');
}

// Check if already installed
if (TokenStorage::isInstalled($shop)) {
    logMessage("App already installed, redirecting to main app", 'INFO', $shop);
    header("Location: index.php?shop=" . urlencode($shop) . "&host=" . urlencode($host));
    exit;
}

// Handle Embedded App Flow (from Partner Dashboard)
if ($id_token) {
    logMessage("Embedded app flow detected with id_token", 'INFO', $shop);
    handleEmbeddedAppInstall($shop, $id_token, $host);
    exit;
}

// Handle Traditional OAuth Callback
if ($code) {
    logMessage("OAuth callback received", 'INFO', $shop);
    handleOAuthCallback($shop, $code, $host);
    exit;
}

// Handle Initial Install Request (Traditional OAuth)
logMessage("Initial install request for shop: $shop", 'INFO', $shop);
initiateOAuthInstall($shop, $host);
exit;

/**
 * Handle Embedded App Installation
 */
function handleEmbeddedAppInstall($shop, $idToken, $host)
{
    $accessToken = exchangeSessionTokenForAccessToken($shop, $idToken);

    if ($accessToken && TokenStorage::storeAccessToken($shop, $accessToken)) {
        logMessage("App installed successfully via embedded flow", 'INFO', $shop);

        $redirectUrl = "index.php?shop=" . urlencode($shop) . "&installed=1";
        if ($host) {
            $redirectUrl .= "&host=" . urlencode($host);
        }
        header("Location: " . $redirectUrl);
        exit;
    } else {
        logMessage("Embedded installation failed", 'ERROR', $shop);
        showError("Installation failed via embedded flow. Please try again.");
    }
}

/**
 * Handle Traditional OAuth Callback
 */
function handleOAuthCallback($shop, $code, $host)
{
    $accessToken = exchangeCodeForAccessToken($shop, $code);

    if ($accessToken && TokenStorage::storeAccessToken($shop, $accessToken)) {
        logMessage("App installed successfully via OAuth", 'INFO', $shop);

        $redirectUrl = "index.php?shop=" . urlencode($shop) . "&installed=1";
        if ($host) {
            $redirectUrl .= "&host=" . urlencode($host);
        }
        header("Location: " . $redirectUrl);
        exit;
    } else {
        logMessage("OAuth installation failed", 'ERROR', $shop);
        showError("Installation failed. Please try again.");
    }
}

/**
 * Initiate Traditional OAuth Install
 */
function initiateOAuthInstall($shop, $host)
{
    $installUrl = generateInstallUrl($shop, $host);
    logMessage("Redirecting to OAuth URL: $installUrl", 'INFO', $shop);
    header("Location: $installUrl");
}

/**
 * Show install form
 */
function showInstallForm()
{
?>
    <!DOCTYPE html>
    <html>

    <head>
        <title>Install Tridentity App</title>
        <link rel="stylesheet" href="assets/style.css">
    </head>

    <body>
        <div class="container">
            <div class="card">
                <h1>Install Tridentity Auto-Login App</h1>
                <p>Enter your Shopify store domain to install the app:</p>

                <form method="GET" action="app.php">
                    <div class="form-group">
                        <label for="shop">Shopify Store Domain:</label>
                        <input type="text" id="shop" name="shop" placeholder="your-store.myshopify.com" required class="form-input">
                    </div>
                    <button type="submit" class="btn btn-primary">Install App</button>
                </form>

                <hr>
                <p>Or install directly for your store:</p>
                <a href="app.php?shop=dy8060-np.myshopify.com" class="btn">Install for dy8060-np.myshopify.com</a>
            </div>
        </div>
    </body>
    </html>
    
<?php
}

/**
 * Show error page
 */
function showError($message)
{
?>
    <!DOCTYPE html>
    <html>

    <head>
        <title>Installation Error</title>
        <link rel="stylesheet" href="assets/style.css">
    </head>

    <body>
        <div class="container">
            <div class="card">
                <div class="alert alert-error">
                    <h2>Installation Failed</h2>
                    <p><?= htmlspecialchars($message) ?></p>
                    <a href="app.php" class="btn">Try Again</a>
                </div>
            </div>
        </div>
    </body>
    </html>
<?php
    exit;
}
?>