<?php

/**
 * Shopify Customer Management
 */

/**
 * Test Shopify API access
 */
function testShopifyAPI($shop, $accessToken)
{
    $url = "https://$shop/admin/api/" . SHOPIFY_API_VERSION . "/shop.json";

    logMessage("Testing Shopify API access for: $shop", 'DEBUG', $shop);

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'X-Shopify-Access-Token: ' . $accessToken,
            'Content-Type: application/json'
        ],
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 30
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    logMessage("Shopify API test - HTTP $httpCode", 'DEBUG', $shop);

    if ($curlError) {
        logMessage("Shopify API test CURL error: $curlError", 'ERROR', $shop);
        return false;
    }

    if ($httpCode === 200) {
        logMessage("Shopify API access confirmed", 'INFO', $shop);
        return true;
    } else {
        logMessage("Shopify API access denied - HTTP $httpCode: $response", 'ERROR', $shop);
        return false;
    }
}

/**
 * Find or create customer in Shopify
 */
function findOrCreateCustomer($shop, $email, $firstName, $lastName, $phone = '')
{
    try {
        logMessage("Finding or creating customer: $email", 'INFO', $shop);

        $accessToken = TokenStorage::getAccessToken($shop);
        if (!$accessToken) {
            logMessage("No access token found for shop: $shop", 'ERROR', $shop);
            return null;
        }

        logMessage("Using access token: " . substr($accessToken, 0, 10) . "...", 'DEBUG', $shop);

        // Test API access first
        if (!testShopifyAPI($shop, $accessToken)) {
            logMessage("Shopify API access denied. Token may be invalid.", 'ERROR', $shop);
            return null;
        }

        // First, try to find existing customer by email
        $customer = findCustomerByEmail($shop, $email, $accessToken);

        if ($customer) {
            logMessage("Found existing customer: " . $customer['id'], 'INFO', $shop);
            return $customer;
        }

        // If not found, create new customer
        logMessage("Creating new customer: $email", 'INFO', $shop);
        return createCustomer($shop, $email, $firstName, $lastName, $phone, $accessToken);
    } catch (Exception $e) {
        logMessage("Error in findOrCreateCustomer: " . $e->getMessage(), 'ERROR', $shop);
        return null;
    }
}

function findCustomerByEmail($shop, $email, $accessToken)
{
    $url = "https://$shop/admin/api/" . SHOPIFY_API_VERSION . "/customers/search.json?query=email:" . urlencode($email);

    logMessage("Searching customer by email: $email", 'DEBUG', $shop);

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'X-Shopify-Access-Token: ' . $accessToken,
            'Content-Type: application/json'
        ],
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 30
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    logMessage("Customer search response: HTTP $httpCode", 'DEBUG', $shop);

    if ($curlError) {
        logMessage("Customer search CURL error: $curlError", 'ERROR', $shop);
        return null;
    }

    if ($httpCode === 200) {
        $data = json_decode($response, true);
        if (!empty($data['customers'])) {
            logMessage("Customer found with ID: " . $data['customers'][0]['id'], 'DEBUG', $shop);
            return $data['customers'][0];
        }
        logMessage("No customer found with email: $email", 'DEBUG', $shop);
        return null;
    } else {
        logMessage("Customer search failed with HTTP $httpCode: $response", 'ERROR', $shop);
        return null;
    }
}

function createCustomer($shop, $email, $firstName, $lastName, $phone, $accessToken)
{
    $url = "https://$shop/admin/api/" . SHOPIFY_API_VERSION . "/customers.json";

    $customerData = [
        'customer' => [
            'email' => $email,
            'first_name' => $firstName,
            'last_name' => $lastName,
            'verified_email' => true,
            'send_email_welcome' => false,
            'tags' => 'tridentity-login'
        ]
    ];

    if (!empty($phone)) {
        $customerData['customer']['phone'] = $phone;
    }

    $postData = json_encode($customerData);

    logMessage("Creating customer with data: " . $postData, 'DEBUG', $shop);

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $postData,
        CURLOPT_HTTPHEADER => [
            'X-Shopify-Access-Token: ' . $accessToken,
            'Content-Type: application/json',
            'Accept: application/json'
        ],
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 30
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    logMessage("Customer creation response: HTTP $httpCode", 'DEBUG', $shop);

    if ($curlError) {
        logMessage("Customer creation CURL error: $curlError", 'ERROR', $shop);
        return null;
    }

    if ($httpCode === 201) {
        $data = json_decode($response, true);
        logMessage("Customer created successfully: " . $data['customer']['id'], 'INFO', $shop);
        return $data['customer'];
    } else {
        logMessage("Customer creation failed with HTTP $httpCode: $response", 'ERROR', $shop);
        return null;
    }
}

/**
 * Generate account activation URL
 */
function generateActivationUrl($shop, $customerId)
{
    $accessToken = TokenStorage::getAccessToken($shop);

    if (!$accessToken) {
        logMessage("No access token available for shop: $shop", 'ERROR', $shop);
        return null;
    }

    $url = "https://$shop/admin/api/" . SHOPIFY_API_VERSION . "/customers/{$customerId}/account_activation_url.json";

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode(['customer' => ['id' => $customerId]]),
        CURLOPT_HTTPHEADER => [
            'X-Shopify-Access-Token: ' . $accessToken,
            'Content-Type: application/json',
            'Accept: application/json'
        ],
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 30
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    logMessage("Activation URL response: HTTP $httpCode", 'DEBUG', $shop);

    if ($curlError) {
        logMessage("Activation URL CURL error: $curlError", 'ERROR', $shop);
        return null;
    }

    if ($httpCode === 200) {
        $data = json_decode($response, true);
        $activationUrl = $data['account_activation_url'] ?? null;

        if ($activationUrl) {
            logMessage("Successfully generated activation URL", 'INFO', $shop);
            return $activationUrl;
        }
    }

    logMessage("Failed to generate activation URL. Response: $response", 'ERROR', $shop);
    return null;
}
