<?php
session_start();
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/shopify-oauth.php'; // Add this line
require_once __DIR__ . '/includes/tridentity-api.php';

$shop = $_GET['shop'] ?? '';
$host = $_GET['host'] ?? '';

if (!$shop) {
    die('Shop parameter is required. Please access this app through the Shopify Admin.');
}

$shop = cleanShopDomain($shop);

// Verify shop domain
if (!verifyShopDomain($shop)) {
    die('Invalid Shopify store domain. Please use: your-store.myshopify.com');
}

// Verify installation and token validity
$isInstalled = TokenStorage::isInstalled($shop);
$accessToken = TokenStorage::getAccessToken($shop);
$tokenValid = false;

if ($isInstalled && $accessToken) {
    // Test if token is still valid
    $url = "https://$shop/admin/api/2024-07/shop.json";
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            'X-Shopify-Access-Token: ' . $accessToken,
            'Content-Type: ' . 'application/json'
        ],
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_TIMEOUT => 10
    ]);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    $tokenValid = ($httpCode === 200);
    
    if (!$tokenValid) {
        logMessage("Stored token is invalid - clearing installation", 'WARNING', $shop);
        TokenStorage::clearInstallation($shop);
        $isInstalled = false;
    }
}

// If not installed or token invalid, redirect to install
if (!$isInstalled || !$tokenValid) {
    logMessage("App not installed or token invalid - redirecting to install", 'INFO', $shop);
    $installUrl = generateInstallUrl($shop, $host);
    header("Location: " . $installUrl);
    exit;
}

// App is properly installed - show the main app interface
logMessage("App loaded successfully for shop: $shop", 'INFO', $shop);

// Handle installation success message
$success_message = '';
if (isset($_GET['installed'])) {
    $success_message = "App installed successfully! You can now test the Tridentity login flow.";
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tridentity Login App</title>
    <link rel="stylesheet" href="assets/style.css">
</head>
<body>
    <div class="app-container">
        <!-- Header -->
        <div class="card">
            <div class="card-header">
                <div class="header-content">
                    <h1>Tridentity Login</h1>
                    <div class="navigation">
                        <a href="index.php?shop=<?= urlencode($shop) ?>" class="nav-btn active">Dashboard</a>
                        <a href="index.php?shop=<?= urlencode($shop) ?>&action=reinstall" class="nav-btn" onclick="return confirm('Reinstall the app? This will clear current settings and generate a new access token.')">Reinstall</a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Success Message -->
        <?php if ($success_message): ?>
            <div class="alert alert-success"><?= htmlspecialchars($success_message) ?></div>
        <?php endif; ?>

        <!-- Main Dashboard -->
        <div class="card">
            <div class="card-body">
                <h2>Welcome to Tridentity Login App</h2>
                <p>Enable seamless customer authentication through Tridentity.</p>

                <div class="status-section">
                    <h3>App Status</h3>
                    <div class="status-grid">
                        <div class="status-item status-success">
                            ✓ App Installed
                        </div>
                        <div class="status-item status-success">
                            ✓ Shopify Connected
                        </div>
                        <div class="status-item status-success">
                            ✓ Tridentity Ready
                        </div>
                    </div>
                </div>

                <div style="margin: 24px 0;">
                    <a href="<?= generateTridentityLoginUrl($shop) ?>" class="btn btn-primary">
                        Test Tridentity Login
                    </a>
                </div>
            </div>
        </div>

        <!-- Setup Instructions -->
        <div class="card">
            <div class="card-body">
                <h3>Setup Instructions</h3>
                <ol class="instructions-list">
                    <li>Click "Test Tridentity Login" to test the complete flow</li>
                    <li>You'll be redirected to Tridentity login page</li>
                    <li>After successful login, you'll be redirected back</li>
                    <li>The system will automatically create/find your customer in Shopify</li>
                    <li>You'll be automatically logged into your Shopify customer account</li>
                </ol>
            </div>
        </div>

        <!-- Integration Code -->
        <div class="card">
            <div class="card-body">
                <h3>Integration Code for Your Theme</h3>
                <p>Add this code to your Shopify theme where you want the login button to appear:</p>

                <div class="code-block">
                    <pre><code>&lt;div class="tridentity-login-section"&gt;
    &lt;a href="<?= generateTridentityLoginUrl($shop) ?>" class="btn btn-tdt-login"&gt;
        Login with Tridentity
    &lt;/a&gt;
&lt;/div&gt;</code></pre>
                </div>
            </div>
        </div>

        <!-- Debug Information -->
        <div class="card">
            <div class="card-body">
                <h3>Debug Information</h3>
                <div class="debug-info">
                    <h4>Connection Status</h4>
                    <div class="debug-grid">
                        <span class="debug-label">Shop:</span>
                        <span class="debug-value"><?= htmlspecialchars($shop) ?></span>
                        
                        <span class="debug-label">Token Status:</span>
                        <span class="debug-value" style="color: #2e5e2e;">✓ Valid</span>
                        
                        <span class="debug-label">Token Preview:</span>
                        <span class="debug-value"><?= substr($accessToken, 0, 20) ?>...</span>
                        
                        <span class="debug-label">App Base URL:</span>
                        <span class="debug-value"><?= APP_BASE_URL ?></span>
                        
                        <span class="debug-label">Callback URL:</span>
                        <span class="debug-value"><?= APP_CALLBACK_URL ?></span>
                        
                        <span class="debug-label">Tridentity App ID:</span>
                        <span class="debug-value"><?= TRIDENTITY_APP_ID ?></span>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        // Handle reinstall action
        const urlParams = new URLSearchParams(window.location.search);
        if (urlParams.get('action') === 'reinstall') {
            if (confirm('Are you sure you want to reinstall the app? This will clear the current access token and require a fresh installation.')) {
                // Redirect to install URL
                window.location.href = '<?= generateInstallUrl($shop, $host) ?>';
            } else {
                // Remove the action parameter from URL
                const newUrl = window.location.pathname + '?shop=<?= urlencode($shop) ?>';
                window.history.replaceState({}, '', newUrl);
            }
        }
    </script>
</body>
</html>